//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.Globalization;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
    /// <summary>
    /// Constructs an object defined through configuration
    /// </summary>
    /// <typeparam name="TObjectToCreate">The object type to create.</typeparam>
    /// <typeparam name="TObjectConfiguration">The object configuration type used to inject data.</typeparam>
    public class ConstructableObject<TObjectToCreate, TObjectConfiguration>
    {
        private readonly Type objectType;
        private readonly TObjectConfiguration objectConfiguration;
        private readonly IConfigurationSource configurationSource;

        /// <summary>
        /// Initialize a new instance of the <see cref="ConstructableObject&lt;TObjectToCreate, TObjectConfiguration&gt;" /> class.
        /// </summary>
        /// <param name="objectType"><see cref="Type"/> of object to create</param>
        /// <param name="objectConfiguration">Configuration information for the object to create</param>
        /// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
        public ConstructableObject(Type objectType, 
                                   TObjectConfiguration objectConfiguration, 
                                   IConfigurationSource configurationSource)
        {
            this.objectType = objectType;
            this.objectConfiguration = objectConfiguration;
            this.configurationSource = configurationSource;
        }
        
        /// <summary>
        /// Creates the object defined by this class
        /// </summary>
        /// <returns>Newly created object of the specified type</returns>
        public TObjectToCreate Create()
        {
            List<ConstructorParameterAttribute> injectionAttributes = GetConstructorInjectionAttributes();

            object[] constructorParameters = GetConstructorParameters(injectionAttributes);
            ConstructorInfo constructor = GetConstructor(injectionAttributes);

            TObjectToCreate result = (TObjectToCreate)constructor.Invoke(constructorParameters);
            return result;
        }

        private object[] GetConstructorParameters(List<ConstructorParameterAttribute> injectionAttributes)
        {
            if (objectConfiguration == null)
            {
                return new object[0];
            }

            object[] parameters = GetParameters(injectionAttributes, objectConfiguration);

            return parameters;
        }

        private ConstructorInfo GetConstructor(List<ConstructorParameterAttribute> attributes)
        {
            Type[] constructorTypes = new Type[attributes.Count];
            int index = 0;
            foreach (ConstructorParameterAttribute attribute in attributes)
            {
                constructorTypes[index++] = attribute.ParameterType;
            }

            ConstructorInfo constructor = objectType.GetConstructor(constructorTypes);
            if (constructor == null)
            {
                StringBuilder signatureStringBuilder = new StringBuilder();
                for (int count = 0; count < constructorTypes.Length; ++count) //each (Type t in constructorTypes)
                {
                    Type t = constructorTypes[count];
                    signatureStringBuilder.Append(t.Name);
                    if (count != (constructorTypes.Length - 1))
                    {
                        signatureStringBuilder.Append(", ");
                    }
                }
                throw new InvalidOperationException(
                    string.Format(Resources.Culture, Resources.ExceptionConstructorNotFound, signatureStringBuilder.ToString(), objectType.Name, this.GetType().Name, objectConfiguration.GetType().Name));
            }

            return constructor;
        }

        private object[] GetParameters(List<ConstructorParameterAttribute> injectionAttributes, TObjectConfiguration objectConfiguration)
        {
            object[] parameters = new object[injectionAttributes.Count];

            for (int i = 0; i < parameters.Length; i++)
            {
                MethodInfo getParameterFactory = typeof(ConstructorParameterAttribute).GetMethod("GetParameterFactory", BindingFlags.NonPublic | BindingFlags.Instance);

                getParameterFactory = getParameterFactory.MakeGenericMethod(injectionAttributes[i].ParameterType, typeof(TObjectConfiguration));

                object parameterProvider = getParameterFactory.Invoke(injectionAttributes[i], new object[] { objectConfiguration });

                Type objectFactory = typeof(IObjectFactory<,>);
                objectFactory = objectFactory.MakeGenericType(injectionAttributes[i].ParameterType, typeof(TObjectConfiguration));

                parameters[i] = objectFactory.InvokeMember("Create",
                        BindingFlags.InvokeMethod | BindingFlags.Public | BindingFlags.Instance,
                        null,
                        parameterProvider,
                        new object[] { objectConfiguration, configurationSource },
                        CultureInfo.CurrentCulture);
            }

            return parameters;
        }

        private List<ConstructorParameterAttribute> GetConstructorInjectionAttributes()
        {
            Attribute[] attributes = Attribute.GetCustomAttributes(objectConfiguration.GetType(), typeof(ConstructorParameterAttribute));
            List<ConstructorParameterAttribute> result = new List<ConstructorParameterAttribute>(attributes.Length);

            Array.Sort<Attribute>(attributes,
                delegate(Attribute x, Attribute y)
                {
                    ConstructorParameterAttribute lhs = (ConstructorParameterAttribute)x;
                    ConstructorParameterAttribute rhs = (ConstructorParameterAttribute)y;

                    return (lhs.Order - rhs.Order);
                });

            int currentIndex = 0;
            foreach (Attribute attribute in attributes)
            {
                ConstructorParameterAttribute injectionAttribute = (ConstructorParameterAttribute)attribute;
                if (injectionAttribute.Order != currentIndex)
                {
                    throw new InvalidOperationException(string.Format(Resources.Culture, Resources.ExceptionInjectionAttributeOrder, injectionAttribute.GetType().FullName, objectConfiguration.GetType().FullName, currentIndex, injectionAttribute.Order));
                }

                result.Add(injectionAttribute);

                currentIndex++;
            }

            return result;
        }
    }
}
